#### READING AND WRITING FROM EXCEL ################################# 
library(tidyverse)




#### OVERVIEW OF PACKAGES ###########################################
# - readxl --------- read from Excel files (Tidyverse)
# - openxlsx ------- write to Excel
# - writexl -------- write to Excel
# - googlesheets4 -- read and write from Google Sheets (Tidyverse)




#### IMPORTING FROM EXCEL ###########################################
# readxl is the way to go
# it's an official tidyverse package :)
library(readxl)

# put this R script in the top level course folder
getwd()
# set your working directory if needed
setwd('R_Data_Analytics_3Day_001/R_Data_Analytics_3Day_001')

# the most general import function is read_excel()
data <- read_excel('data/example_excel_data.xlsx')
data

# but if you know the file extension, use the more specific import versions
# use read_xlsx() or read_xls()
data <- read_xlsx('data/example_excel_data.xlsx')
data

# what sheets are in that notebook?
excel_sheets('data/example_excel_data.xlsx')

# let's open one of the other sheets
gss_vocab <- read_xlsx('data/example_excel_data.xlsx', sheet = 'GSS Vocab')
gss_vocab %>% head()

# or define sheet by number
transactions <- read_xlsx('data/example_excel_data.xlsx', sheet = 3)
transactions %>% head()

# notice the difference between CSV and Excel files here
# the [TransactionDate] column is a good example
# the data type is preserved from Excel ---- datetime
class(transactions$TransactionDate)

# there are different ways to specify the range of sheets that are imported
read_xlsx('data/example_excel_data.xlsx', sheet = 2, range = 'C1:E4')
read_xlsx('data/example_excel_data.xlsx', sheet = 2, range = cell_rows(1:4))
read_xlsx('data/example_excel_data.xlsx', sheet = 2, range = cell_cols(1:4))
read_xlsx('data/example_excel_data.xlsx', range = 'Transactions!C1:E4')

# see more at the documentation:
# https://readxl.tidyverse.org/
  



#### WRITING TO EXCEL ###############################################
# there are two main options for writing from R to Excel:
# - openxlsx
# - writexl
# openxlsx is larger and more comprehensive
# writexl is smaller, has less features, but appears to write faster

# vignettes for openxlsx
# - introduction:
#   - https://cran.r-project.org/web/packages/openxlsx/vignettes/Introduction.html
# - styling your Excel sheets using R
#   - https://cran.r-project.org/web/packages/openxlsx/vignettes/Formatting.html




#### WRITING TO EXCEL DEMO ##########################################
# this is from the Introductory vignette for openxlsx
library(openxlsx)

wb <- createWorkbook()

## read historical prices from yahoo finance
ticker <- "CBA.AX"
csv.url <- paste0(
  "https://query1.finance.yahoo.com/v7/finance/download/", 
  ticker,
  "?period1=1597597610&period2=1629133610&interval=1d&events=history&includeAdjustedClose= TRue"
)
prices <- read.csv(url(csv.url), as.is = TRUE)
prices$Date <- as.Date(prices$Date)
close <- prices$Close
prices$logReturns = c(0, log(close[2:length(close)]/close[1:(length(close) - 1)]))

## Create plot of price series and add to worksheet
ggplot(data = prices, aes(as.Date(Date), as.numeric(Close))) + 
  geom_line(colour = "royalblue2") +
  labs(x = "Date", y = "Price", title = ticker) + 
  geom_area(fill = "royalblue1", alpha = 0.3) + 
  coord_cartesian(ylim = c(min(prices$Close) - 1.5, max(prices$Close) + 1.5))

## Add worksheet and write plot to sheet
addWorksheet(wb, sheetName = "CBA")
insertPlot(wb, sheet = 1, xy = c("J", 3))

## Histogram of log returns
ggplot(data = prices, aes(x = logReturns)) + 
  geom_histogram(binwidth = 0.0025) +
  labs(title = "Histogram of log returns")

## currency
class(prices$Close) <- "currency"  ## styles as currency in workbook

## write historical data and histogram of returns
writeDataTable(wb, sheet = "CBA", x = prices)
insertPlot(wb, sheet = 1, startRow = 25, startCol = "J")

## Add conditional formatting to show where logReturn > 0.01 using default
## style
conditionalFormatting(
  wb, 
  sheet = 1, 
  cols = 1:ncol(prices), 
  rows = 2:(nrow(prices) + 1), 
  rule = "$H2 > 0.01"
)

## style log return col as a percentage
logRetStyle <- createStyle(numFmt = "percentage")
addStyle(
  wb, 
  sheet = 1, 
  style = logRetStyle, 
  rows = 2:(nrow(prices) + 1), 
  cols = "H", 
  gridExpand = TRUE
)
setColWidths(
  wb, 
  sheet = 1, 
  cols = c("A", "F", "G", "H"), 
  widths = 15
)

## save workbook to working directory
saveWorkbook(wb, "stockPrice.xlsx", overwrite = TRUE)
openXL("stockPrice.xlsx")




#### googlesheets4 ##################################################
# for reading and writing from Google Sheets, use googlesheets4 package
# it's an official tidyverse package
# https://googlesheets4.tidyverse.org/