# Task 1: Navigating Directories

# Get current location
Write-Host "Current Directory:"
Get-Location

# Navigate to C:\Users (or home directory on macOS)
Write-Host "Navigating to C:\Users (or ~/ on macOS)..."
if ($IsWindows) {
    Set-Location C:\Users
} else {
    Set-Location ~
}

# Return to previous directory
Write-Host "Returning to the previous directory..."
Set-Location -

# List contents of C:\Windows (or / on macOS)
Write-Host "Listing contents of C:\Windows (or / on macOS)..."
if ($IsWindows) {
    Get-ChildItem C:\Windows
} else {
    Get-ChildItem /
}

# Task 2: Creating and Deleting Directories

# Create a new directory named TestFolder
Write-Host "Creating 'TestFolder'..."
New-Item -Path "$HOME\TestFolder" -ItemType Directory

# Verify creation
Write-Host "Verifying 'TestFolder' exists..."
Get-ChildItem -Path "$HOME\TestFolder"

# Create two subdirectories
Write-Host "Creating 'Sub1' and 'Sub2' inside 'TestFolder'..."
New-Item -Path "$HOME\TestFolder\Sub1" -ItemType Directory
New-Item -Path "$HOME\TestFolder\Sub2" -ItemType Directory

# Remove Sub1
Write-Host "Deleting 'Sub1'..."
Remove-Item -Path "$HOME\TestFolder\Sub1"

# Attempt to remove TestFolder without -Recurse (expected error)
Write-Host "Attempting to delete 'TestFolder' without -Recurse..."
try {
    Remove-Item -Path "$HOME\TestFolder"
} catch {
    Write-Host "Error encountered: $_"
}

# Remove TestFolder with -Recurse
Write-Host "Deleting 'TestFolder' and all its contents..."
Remove-Item -Path "$HOME\TestFolder" -Recurse -Force

# Task 3: Reading and Writing Files

# Create and write to a file
$filePath = "$HOME\example.txt"
Write-Host "Creating 'example.txt' and writing 'Hello, PowerShell!'..."
Set-Content -Path $filePath -Value "Hello, PowerShell!"

# Append a line
Write-Host "Appending 'This is an additional line.'..."
Add-Content -Path $filePath -Value "This is an additional line."

# Read and display file contents
Write-Host "Reading contents of 'example.txt'..."
Get-Content -Path $filePath

# Modify file by adding another line
Write-Host "Adding 'This is a modification.'..."
$content = Get-Content -Path $filePath
$content += "This is a modification."
Set-Content -Path $filePath -Value $content

# Task 4: Managing File and Directory Permissions

if ($IsWindows) {
    # Windows ACL Management
    Write-Host "Checking permissions of 'example.txt'..."
    Get-Acl -Path $filePath

    Write-Host "Modifying file owner (admin privileges required)..."
    $acl = Get-Acl -Path $filePath
    $acl.SetOwner([System.Security.Principal.NTAccount]"$env:USERNAME")
    Set-Acl -Path $filePath -AclObject $acl

    Write-Host "Granting Read permission to another user..."
    $rule = New-Object System.Security.AccessControl.FileSystemAccessRule("$env:USERNAME", "Read", "Allow")
    $acl.SetAccessRule($rule)
    Set-Acl -Path $filePath -AclObject $acl

    Write-Host "Revoking the added permission..."
    $acl.RemoveAccessRule($rule)
    Set-Acl -Path $filePath -AclObject $acl
} else {
    # macOS/Linux Permission Management
    Write-Host "Checking file permissions using ls -l..."
    ls -l $filePath

    Write-Host "Granting read/write permissions to all users..."
    chmod 666 $filePath

    Write-Host "Changing file ownership (admin privileges required)..."
    sudo chown $env:USER $filePath

    Write-Host "Restoring original permissions..."
    chmod 644 $filePath
}

Write-Host "Script execution completed."