CREATE OR REPLACE TRIGGER employees_compound_trigger
FOR INSERT OR UPDATE OR DELETE ON employees
COMPOUND TRIGGER

  l_min_salary jobs.min_salary%TYPE;
  l_max_salary jobs.max_salary%TYPE;
  l_managers_manager_id   employees.manager_id%TYPE;
  l_operation CHAR := CASE 
      WHEN UPDATING THEN 'U'
      WHEN DELETING THEN 'D'
      ELSE 'I' END;
  invalid_salary EXCEPTION;
  num_head_honchos        NUMBER(6,0);
  cannot_remove_boss      EXCEPTION;

BEFORE EACH ROW IS
BEGIN

  SELECT min_salary, max_salary
  INTO l_min_salary, l_max_salary
  FROM jobs
  WHERE job_id = :NEW.job_id;
  
  IF :NEW.salary NOT BETWEEN l_min_salary AND l_max_salary THEN
    RAISE invalid_salary;
  END IF;
  
EXCEPTION
  WHEN invalid_salary THEN
    -- re-raise the error to be caught by calling code
    RAISE_APPLICATION_ERROR(-20101,
              'Salary for ' || :NEW.job_id || ' must be between ' ||
              l_min_salary || ' and ' || l_max_salary || '.');
END BEFORE EACH ROW;

AFTER EACH ROW IS
BEGIN
  IF INSERTING OR UPDATING THEN
    INSERT INTO employees_audit (
      employee_id,
      first_name,
      last_name,
      email,
      phone_number,
      hire_date,
      job_id,
      salary,
      commission_pct,
      manager_id,
      department_id,
      operation,
      change_date,
      change_made_by
    )
    VALUES (
      :NEW.employee_id,
      :NEW.first_name,
      :NEW.last_name,
      :NEW.email,
      :NEW.phone_number,
      :NEW.hire_date,
      :NEW.job_id,
      :NEW.salary,
      :NEW.commission_pct,
      :NEW.manager_id,
      :NEW.department_id,
      l_operation,
      SYSDATE,
      USER
    );
  ELSE -- DELETING
    INSERT INTO employees_audit (
      employee_id,
      first_name,
      last_name,
      email,
      phone_number,
      hire_date,
      job_id,
      salary,
      commission_pct,
      manager_id,
      department_id,
      operation,
      change_date,
      change_made_by
    )
    VALUES (
      :OLD.employee_id,
      :OLD.first_name,
      :OLD.last_name,
      :OLD.email,
      :OLD.phone_number,
      :OLD.hire_date,
      :OLD.job_id,
      :OLD.salary,
      :OLD.commission_pct,
      :OLD.manager_id,
      :OLD.department_id,
      l_operation,
      SYSDATE,
      USER
    );
  END IF;
END AFTER EACH ROW;

AFTER STATEMENT IS 
BEGIN
  SELECT COUNT(*)
  INTO num_head_honchos
  FROM employees
  WHERE manager_id IS NULL;
  
  IF num_head_honchos = 0 THEN
    RAISE cannot_remove_boss;
  END IF;
  
EXCEPTION
  WHEN cannot_remove_boss THEN
    RAISE_APPLICATION_ERROR(-20101,
                            'Must have at least one head honcho.');
END AFTER STATEMENT;

END employees_compound_trigger;