CREATE OR REPLACE PACKAGE BODY employee_package
IS
  /* PRIVATE */
  -- AUDITING
  PROCEDURE add_job_history(
    employee_id_in    job_history.employee_id%TYPE,
    start_date_in     job_history.start_date%TYPE,
    end_date_in       job_history.end_date%TYPE,
    job_id_in         job_history.job_id%TYPE,
    department_id_in  job_history.department_id%TYPE
  )
  IS
  BEGIN
    
    INSERT INTO job_history
    (employee_id, start_date, end_date, job_id, department_id)
    VALUES( employee_id_in, start_date_in, 
            end_date_in, job_id_in, department_id_in);
  END add_job_history;

  /* PUBLIC */
  -- INSERT
  FUNCTION new_employee(
        first_name_in       IN employees.first_name%TYPE,
        last_name_in        IN employees.last_name%TYPE,
        email_in            IN employees.email%TYPE,
        phone_number_in     IN employees.phone_number%TYPE,
        job_id_in           IN employees.job_id%TYPE,
        salary_in           IN employees.salary%TYPE,
        manager_id_in       IN employees.manager_id%TYPE,
        department_id_in    IN employees.department_id%TYPE,
        commission_pct_in   IN employees.commission_pct%TYPE := NULL,
        hire_date_in        IN employees.hire_date%TYPE := SYSDATE
    )
    RETURN employees%ROWTYPE
  IS
    l_employee_id employees.employee_id%TYPE;
    l_employee    employees%ROWTYPE;
  BEGIN
    -- Get new employee_id
    SELECT MAX(employee_id) + 1
    INTO l_employee_id
    FROM employees;
  
    INSERT INTO employees (employee_id, first_name, last_name,
                          email, phone_number, hire_date,
                          job_id, salary, commission_pct,
                          manager_id, department_id)
    VALUES               (l_employee_id, first_name_in, last_name_in,
                          email_in, phone_number_in, hire_date_in,
                          job_id_in, salary_in, commission_pct_in,
                          manager_id_in, department_id_in)
    RETURNING employee_id, first_name, last_name,
              email, phone_number, hire_date,
              job_id, salary, commission_pct,
              manager_id, department_id INTO l_employee;
    
    RETURN l_employee;
  
  END new_employee;
  
  -- GETTERS (RECORDS)
  FUNCTION get_employee(employee_id_in IN employees.employee_id%TYPE)
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE employee_id = employee_id_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_employee(email_in IN employees.email%TYPE) 
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE email = email_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_employee(first_name_in   IN employees.first_name%TYPE,
                        last_name_in    IN employees.last_name%TYPE) 
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE first_name = first_name_in
      AND last_name = last_name_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_manager(employee_id_in IN employees.employee_id%TYPE)
    RETURN employees%ROWTYPE
  IS
    l_employee        employees%ROWTYPE;
    l_manager         employees%ROWTYPE;
    no_such_employee  EXCEPTION;
  BEGIN
  
    l_employee := get_employee(employee_id_in);
    IF l_employee.employee_id IS NULL THEN
      RAISE no_such_employee;
    END IF;
    
    SELECT *
    INTO l_manager
    FROM employees
    WHERE employee_id = (
      SELECT manager_id
      FROM employees
      WHERE employee_id = employee_id_in);
  
    RETURN l_manager;
  
  EXCEPTION
    WHEN no_such_employee THEN
      RAISE_APPLICATION_ERROR(-20102,
                              'No such employee.');
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_manager;
  
  -- GETTERS (FIELDS)  
  FUNCTION get_hire_date(
                  employee_id_in IN employees.employee_id%TYPE
  )
    RETURN employees.hire_date%TYPE
  IS
    l_hire_date employees.hire_date%TYPE;
  BEGIN
    SELECT hire_date
    INTO l_hire_date
    FROM employees
    WHERE employee_id = employee_id_in;
  
    RETURN l_hire_date;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_hire_date;
  
  -- SETTERS (UPDATES)
  PROCEDURE change_job(
    employee_id_in    employees.employee_id%TYPE,
    job_id_in         employees.job_id%TYPE,
    department_id_in  employees.department_id%TYPE
  )
  IS
    l_old_job_id          job_history.job_id%TYPE;
    l_old_department_id   job_history.department_id%TYPE;
    l_start_date          job_history.start_date%TYPE;
    l_end_date            job_history.end_date%TYPE := SYSDATE;
  BEGIN
  
    -- Store old job_id and department_id in local variables
    SELECT job_id, department_id
    INTO l_old_job_id, l_old_department_id
    FROM employees
    WHERE employee_id = employee_id_in;
    
    -- For start date of old job get day after end date of prior job
    SELECT MAX(end_date) + 1 -- Plus 1 day
    INTO l_start_date
    FROM job_history
    WHERE employee_id = employee_id_in;
    
    -- If this is first job change,
    -- get hire date for start date of old job
    IF l_start_date IS NULL THEN
      SELECT hire_date
      INTO l_start_date
      FROM employees
      WHERE employee_id = employee_id_in;
    END IF;
    
    add_job_history(employee_id_in    => employee_id_in,
                    start_date_in     => l_start_date,
                    end_date_in       => l_end_date,
                    job_id_in         => l_old_job_id,
                    department_id_in  => l_old_department_id);
  
    UPDATE employees
    SET job_id = job_id_in,
        department_id = department_id_in
    WHERE employee_id = employee_id_in;
  
  END change_job;
    
  PROCEDURE change_manager(
    employee_id_in      IN    employees.employee_id%TYPE,
    manager_id_in       IN    employees.manager_id%TYPE
  ) IS
  BEGIN
    check_manager(employee_id_in, manager_id_in);
    
    -- Set new manager
    UPDATE employees
    SET manager_id = manager_id_in
    WHERE employee_id = employee_id_in;
  END change_manager;
  
  -- VALIDATION
  PROCEDURE check_manager(
    employee_id_in  IN      employees.employee_id%TYPE,
    manager_id_in   IN      employees.manager_id%TYPE
  )
  IS
    l_managers_manager_id employees.manager_id%TYPE := -1;
    l_temp_employee_id    employees.employee_id%TYPE := manager_id_in;
    manager_is_self       EXCEPTION;
    reciprocal_managers   EXCEPTION;
  BEGIN
    IF employee_id_in = manager_id_in THEN
      RAISE manager_is_self;
    END IF;
    
    WHILE l_managers_manager_id IS NOT NULL LOOP
      SELECT manager_id
      INTO l_managers_manager_id
      FROM employees
      WHERE employee_id = l_temp_employee_id;
      
      IF l_managers_manager_id = employee_id_in THEN
        RAISE reciprocal_managers;
      END IF;
      l_temp_employee_id := l_managers_manager_id;
      
    END LOOP;
  -- Made it through loop without finding match
    
  EXCEPTION
    WHEN manager_is_self THEN
      -- re-raise the error to be caught by calling code
      RAISE_APPLICATION_ERROR(-20101,
                              'Employee cannot be own manager.');
    WHEN reciprocal_managers THEN
      -- re-raise the error to be caught by calling code
      RAISE_APPLICATION_ERROR(-20102,
                        'Employees cannot report to underlings.');
    
  END check_manager;

  -- DELETE
  FUNCTION delete_employee(
                  employee_id_in IN employees.employee_id%TYPE
  )
    RETURN BOOLEAN
  IS
  BEGIN
    DELETE
    FROM employees
    WHERE employee_id = employee_id_in;
    
    RETURN SQL%FOUND;
  
  END delete_employee;
  
END employee_package;