CREATE OR REPLACE PACKAGE BODY employee_package
IS
  -- INSERT
  FUNCTION new_employee(
        first_name_in       IN employees.first_name%TYPE,
        last_name_in        IN employees.last_name%TYPE,
        email_in            IN employees.email%TYPE,
        phone_number_in     IN employees.phone_number%TYPE,
        job_id_in           IN employees.job_id%TYPE,
        salary_in           IN employees.salary%TYPE,
        manager_id_in       IN employees.manager_id%TYPE,
        department_id_in    IN employees.department_id%TYPE,
        commission_pct_in   IN employees.commission_pct%TYPE := NULL,
        hire_date_in        IN employees.hire_date%TYPE := SYSDATE
    )
    RETURN employees%ROWTYPE
  IS
    l_employee_id employees.employee_id%TYPE;
    l_employee    employees%ROWTYPE;
  BEGIN
    -- Get new employee_id
    SELECT MAX(employee_id) + 1
    INTO l_employee_id
    FROM employees;
  
    INSERT INTO employees (employee_id, first_name, last_name,
                          email, phone_number, hire_date,
                          job_id, salary, commission_pct,
                          manager_id, department_id)
    VALUES               (l_employee_id, first_name_in, last_name_in,
                          email_in, phone_number_in, hire_date_in,
                          job_id_in, salary_in, commission_pct_in,
                          manager_id_in, department_id_in)
    RETURNING employee_id, first_name, last_name,
              email, phone_number, hire_date,
              job_id, salary, commission_pct,
              manager_id, department_id INTO l_employee;
    
    RETURN l_employee;
  
  END new_employee;
  
  -- GETTERS (RECORDS)
  FUNCTION get_employee(employee_id_in IN employees.employee_id%TYPE)
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE employee_id = employee_id_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_employee(email_in IN employees.email%TYPE) 
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE email = email_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_employee(first_name_in   IN employees.first_name%TYPE,
                        last_name_in    IN employees.last_name%TYPE) 
    RETURN employees%ROWTYPE
  IS
    l_employee employees%ROWTYPE;
  BEGIN
    SELECT *
    INTO l_employee
    FROM employees
    WHERE first_name = first_name_in
      AND last_name = last_name_in;
  
    RETURN l_employee;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_employee;
  
  FUNCTION get_manager(employee_id_in IN employees.employee_id%TYPE)
    RETURN employees%ROWTYPE
  IS
    l_employee        employees%ROWTYPE;
    l_manager         employees%ROWTYPE;
    no_such_employee  EXCEPTION;
  BEGIN
  
    l_employee := get_employee(employee_id_in);
    IF l_employee.employee_id IS NULL THEN
      RAISE no_such_employee;
    END IF;
    
    SELECT *
    INTO l_manager
    FROM employees
    WHERE employee_id = (
      SELECT manager_id
      FROM employees
      WHERE employee_id = employee_id_in);
  
    RETURN l_manager;
  
  EXCEPTION
    WHEN no_such_employee THEN
      RAISE_APPLICATION_ERROR(-20102,
                              'No such employee.');
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_manager;
  
  -- GETTERS (FIELDS)  
  FUNCTION get_hire_date(
                  employee_id_in IN employees.employee_id%TYPE
  )
    RETURN employees.hire_date%TYPE
  IS
    l_hire_date employees.hire_date%TYPE;
  BEGIN
    SELECT hire_date
    INTO l_hire_date
    FROM employees
    WHERE employee_id = employee_id_in;
  
    RETURN l_hire_date;
  
  EXCEPTION
    WHEN no_data_found THEN
      RETURN NULL;
  
  END get_hire_date;

  -- DELETE
  FUNCTION delete_employee(
                  employee_id_in IN employees.employee_id%TYPE
  )
    RETURN BOOLEAN
  IS
  BEGIN
    DELETE
    FROM employees
    WHERE employee_id = employee_id_in;
    
    RETURN SQL%FOUND;
  
  END delete_employee;
  
END employee_package;