-- Create the procedure
CREATE OR REPLACE PROCEDURE replace_head_honcho(
  new_head_honcho_id_in      IN    employees.employee_id%TYPE,
  old_head_honcho_id_in      IN    employees.employee_id%TYPE,
  new_head_honcho_name_out   OUT   VARCHAR2,
  old_head_honcho_name_out   OUT   VARCHAR2
) IS
  l_new_head_manager_id         employees.manager_id%TYPE;
  l_old_head_manager_id         employees.manager_id%TYPE;
  employee_already_head_honcho  EXCEPTION;
  employee_not_head_honcho      EXCEPTION;
BEGIN
  
  SELECT manager_id
  INTO l_new_head_manager_id
  FROM employees
  WHERE employee_id = new_head_honcho_id_in;
  
  SELECT manager_id
  INTO l_old_head_manager_id
  FROM employees
  WHERE employee_id = old_head_honcho_id_in;
  
  SELECT first_name || ' ' || last_name
  INTO new_head_honcho_name_out
  FROM employees
  WHERE employee_id = new_head_honcho_id_in;
  
  SELECT first_name || ' ' || last_name
  INTO old_head_honcho_name_out
  FROM employees
  WHERE employee_id = old_head_honcho_id_in;
  
  CASE
    WHEN l_new_head_manager_id IS NULL THEN
      RAISE employee_already_head_honcho;
    WHEN l_old_head_manager_id IS NOT NULL THEN
      RAISE employee_not_head_honcho;
    ELSE
      -- New head honcho should not report to anyone
      UPDATE employees
      SET manager_id = NULL
      WHERE employee_id = new_head_honcho_id_in;
      
      -- Old head honcho should report to new head honcho
      UPDATE employees
      SET manager_id = new_head_honcho_id_in
      WHERE employee_id = old_head_honcho_id_in;
  END CASE;
EXCEPTION
  WHEN employee_already_head_honcho THEN
    RAISE_APPLICATION_ERROR(-20101,
              new_head_honcho_name_out ||
              '(employee_id: ' || new_head_honcho_id_in ||
              ') is already a head honcho.');
  WHEN employee_not_head_honcho THEN
    RAISE_APPLICATION_ERROR(-20102,
              old_head_honcho_name_out ||
              '(employee_id: ' || old_head_honcho_id_in ||
              ') is not a head honcho.');
  
END replace_head_honcho;

-- Call the procedure
DECLARE
  l_new_head_honcho_id    employees.employee_id%TYPE  :=  102;
  l_old_head_honcho_id    employees.manager_id%TYPE   :=  100;
  l_new_head_honcho_name  VARCHAR2(50);
  l_old_head_honcho_name  VARCHAR2(50);
BEGIN
  replace_head_honcho(
    new_head_honcho_id_in     => l_new_head_honcho_id,
    old_head_honcho_id_in     => l_old_head_honcho_id,
    new_head_honcho_name_out  => l_new_head_honcho_name,
    old_head_honcho_name_out  => l_old_head_honcho_name
  );
  
  DBMS_OUTPUT.PUT_LINE(l_old_head_honcho_name ||
                    ' now reports to ' ||
                    l_new_head_honcho_name || '.');
  
  DBMS_OUTPUT.PUT_LINE(l_new_head_honcho_name ||
                    ' is new head honcho.');
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20001, 'An error was encountered - ' ||
                                    SQLCODE ||
                                    ' -ERROR- ' ||
                                    SQLERRM, TRUE);
END;