-- Create the procedure
CREATE OR REPLACE PROCEDURE update_employee_manager(
  employee_id_in    IN  employees.employee_id%TYPE,
  new_manager_id_in IN  employees.manager_id%TYPE
  /*  STEP 1
    Add OUT parameters:
      - employee_name_out - employee's full name
      - old_manager_name_out - old manager's full name
      - new_manager_name_out - new manager's full name
  */
) IS
  manager_is_self      EXCEPTION;
  /*  STEP 2
    Declare new variables
      - l_old_manager_id - to hold old manager's employee_id.
                           Should default to 0.
      - l_new_managers_manager_id - to hold new manager's manager_id.
                                    Should default to 0.
      - l_num_head_honchos - to hold the number of employees who
                             have no manager assigned
                             
    Declare two new EXCEPTION variables:
      - cannot_remove_last_head_honcho
      - reciprocal_managers
  */
BEGIN
  IF employee_id_in = new_manager_id_in THEN
    RAISE manager_is_self;
  END IF;
  
  /*  STEP 3
    Select the employee's current manager_id and full name
    into l_old_manager_id and employee_name_out, respectively.
  */
  
    -- Get name of old manager
  IF l_old_manager_id IS NOT NULL THEN
    SELECT first_name || ' ' || last_name
    INTO old_manager_name_out
    FROM employees
    WHERE employee_id = l_old_manager_id;
  END IF;
  
  /*  STEP 4
    If the passed-in new_manager_id_in is not NULL, it means that the
    employee is being assigned a new manager. Select this new
    manager's manager_id and full name into l_new_managers_manager_id
    and new_manager_name_out, respectively. We need to know the new
    manager's manager_id so that we can avoid a reciprocal
    employee-manager relationship.
  */
  
  /* STEP 5
    If l_old_manager_id is not NULL, it means that the employee
    had a previous manager. Select that manager's name into
    old_manager_name_out. This step is already done:
  */

  /*  STEP 6
    If new_manager_id_in is not NULL, that means we are assigning a new
    manager to this employee. We have two concerns:
    1.  That manager cannot be someone who reports to the employee.
    2.  If this employee is the only head honcho, we can't assign them
        a manager as that would leave us with no head honchos.
    These checks have been added already. You just need to raise the
    appropriate exceptions.
  */
  IF new_manager_id_in IS NOT NULL THEN
    -- Get manager id and full name of the passed-in manager
    SELECT manager_id, first_name || ' ' || last_name
    INTO l_new_managers_manager_id, new_manager_name_out
    FROM employees
    WHERE employee_id = new_manager_id_in;

    /* STEP 6a
      Raise a reciprocal_managers exception if employee_id_in is
      equal to l_new_managers_manager_id.
    */

    -- Get number of head honchos not including this employee
    -- This will be the number of head honchos left after assigning
    -- this employee a manager.
    SELECT COUNT(employee_id)
    INTO l_num_head_honchos
    FROM employees
    WHERE manager_id IS NULL AND employee_id <> employee_id_in;

    /* STEP 6b
      Raise a cannot_remove_last_head_honcho exception if
      l_num_head_honchos = 0.
    */
  END IF;
  
  -- Set new manager
  UPDATE employees
  SET manager_id = new_manager_id_in
  WHERE employee_id = employee_id_in;
  
EXCEPTION
  WHEN manager_is_self THEN
    RAISE_APPLICATION_ERROR(-20101,
                            'Employee cannot be own manager.');
   /*  STEP 7
    When there is a reciprocal_managers exception, re-raise
    the exception with an appropriate error code and message.
  */

  /*  STEP 8
    When there is a cannot_remove_last_head_honcho exception, re-raise
    the exception with an appropriate error code and message.
  */
END update_employee_manager;

-- Call the procedure
DECLARE
  l_employee_id       employees.employee_id%TYPE  :=  204;
  l_new_manager_id    employees.manager_id%TYPE   :=  100;
  l_employee_name     VARCHAR2(50);
  l_old_manager_name  VARCHAR2(50);
  l_new_manager_name  VARCHAR2(50);
BEGIN
  update_employee_manager(
    employee_id_in        => l_employee_id,
    new_manager_id_in     => l_new_manager_id,
    employee_name_out     => l_employee_name,
    old_manager_name_out  => l_old_manager_name,
    new_manager_name_out  => l_new_manager_name
  );
  DBMS_OUTPUT.PUT_LINE('Employee:    ' || l_employee_name);
  DBMS_OUTPUT.PUT_LINE('Old Manager: ' || l_old_manager_name);
  DBMS_OUTPUT.PUT_LINE('New Manager: ' || l_new_manager_name);
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20001, 'An error was encountered - ' ||
                                    SQLCODE ||
                                    ' -ERROR- ' ||
                                    SQLERRM, TRUE);
END;