-- Create the procedure
CREATE OR REPLACE PROCEDURE update_employee_manager(
  employee_id_in        IN    employees.employee_id%TYPE,
  new_manager_id_in     IN    employees.manager_id%TYPE,
  employee_name_out     OUT   VARCHAR2,
  old_manager_name_out  OUT   VARCHAR2,
  new_manager_name_out  OUT   VARCHAR2
) IS
  l_old_manager_id                employees.manager_id%TYPE := 0;
  l_new_managers_manager_id       employees.manager_id%TYPE := 0;
  l_num_head_honchos              NUMBER(6,0);
  manager_is_self                 EXCEPTION;
  cannot_remove_last_head_honcho  EXCEPTION;
  reciprocal_managers             EXCEPTION;

  -- Default l_temp_employee_id to new_manager_id_in. Used to check
  -- that employee_id_in doesn't report to l_temp_employee_id's boss
  l_temp_employee_id employees.employee_id%TYPE := new_manager_id_in;
BEGIN
  IF employee_id_in = new_manager_id_in THEN
    RAISE manager_is_self;
  END IF;
  
  -- Get manager id and name of employee
  SELECT manager_id, first_name || ' ' || last_name
  INTO l_old_manager_id, employee_name_out
  FROM employees
  WHERE employee_id = employee_id_in;

  -- Get name of old manager
  IF l_old_manager_id IS NOT NULL THEN
    SELECT first_name || ' ' || last_name
    INTO old_manager_name_out
    FROM employees
    WHERE employee_id = l_old_manager_id;
  END IF;
  
  IF new_manager_id_in IS NOT NULL THEN
    -- Get full name of the passed-in manager
    SELECT first_name || ' ' || last_name
    INTO new_manager_name_out
    FROM employees
    WHERE employee_id = new_manager_id_in;

    -- Make sure employee is not reporting to an underling
    WHILE l_new_managers_manager_id IS NOT NULL LOOP
      SELECT manager_id
      INTO l_new_managers_manager_id
      FROM employees
      WHERE employee_id = l_temp_employee_id;
      
      DBMS_OUTPUT.PUT_LINE(l_temp_employee_id || ' reports to ' ||
                          l_new_managers_manager_id);
      
      IF l_new_managers_manager_id = employee_id_in THEN
        RAISE reciprocal_managers;
      END IF;
      -- Assign manager's manager id to l_temp_employee_id
      l_temp_employee_id := l_new_managers_manager_id;
      
    END LOOP;

    -- Get number of head honchos not including this employee
    -- This will be the number of head honchos left after assigning
    -- this employee a manager.
    SELECT COUNT(employee_id)
    INTO l_num_head_honchos
    FROM employees
    WHERE manager_id IS NULL AND employee_id <> employee_id_in;

    -- Cannot remove last head honcho
    IF l_num_head_honchos = 0 THEN
      RAISE cannot_remove_last_head_honcho;
    END IF;
  END IF;

  -- Set new manager
  UPDATE employees
  SET manager_id = new_manager_id_in
  WHERE employee_id = employee_id_in;
  
EXCEPTION
  WHEN manager_is_self THEN
    RAISE_APPLICATION_ERROR(-20101,
                            'Employee cannot be own manager.');
  WHEN reciprocal_managers THEN
    RAISE_APPLICATION_ERROR(-20102,
                      'Employee cannot be managed by an underling.');
  WHEN cannot_remove_last_head_honcho THEN
    RAISE_APPLICATION_ERROR(-20103,
                            'Cannot remove last head honcho.');
END update_employee_manager;

-- Call the procedure
DECLARE
  l_employee_id       employees.employee_id%TYPE  :=  102;
  l_new_manager_id    employees.manager_id%TYPE   :=  107;
  l_employee_name     VARCHAR2(50);
  l_old_manager_name  VARCHAR2(50);
  l_new_manager_name  VARCHAR2(50);
BEGIN
  update_employee_manager(
    employee_id_in        => l_employee_id,
    new_manager_id_in     => l_new_manager_id,
    employee_name_out     => l_employee_name,
    old_manager_name_out  => l_old_manager_name,
    new_manager_name_out  => l_new_manager_name
  );
  DBMS_OUTPUT.PUT_LINE('Employee:    ' || l_employee_name);
  DBMS_OUTPUT.PUT_LINE('Old Manager: ' || l_old_manager_name);
  DBMS_OUTPUT.PUT_LINE('New Manager: ' || l_new_manager_name);
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20001, 'An error was encountered - ' ||
                                    SQLCODE ||
                                    ' -ERROR- ' ||
                                    SQLERRM, TRUE);
END;