CREATE OR REPLACE PROCEDURE insert_person(
  first_name_in     IN    people.first_name%TYPE,
  last_name_in      IN    people.last_name%TYPE,
  email_in          IN    people.email%TYPE,
  phone_number_in   IN    people.phone_number%TYPE,
  company_name_in   IN    companies.company_name%TYPE,
  company_type_in   IN    VARCHAR2, -- staff, client, or vendor
  hire_date_in      IN    DATE :=  SYSDATE
) IS
  l_company_id          companies.company_id%TYPE;
  l_staff               companies.staff%TYPE  := '0';
  l_client              companies.client%TYPE := '0';
  l_vendor              companies.vendor%TYPE := '0';
  l_staff_person_id     people.person_id%TYPE; -- in case we insert a staff person
  no_such_company_type  EXCEPTION;
BEGIN
  -- Save current values into variable if company already in table.
  -- If it doesn't (NO_DATA_FOUND), fail silently.
  BEGIN
    SELECT company_id, staff, client, vendor
    INTO l_company_id, l_staff, l_client, l_vendor
    FROM companies
    WHERE company_name = company_name_in;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      NULL;
  END;
  
  -- Change values of l_staff, l_client, and l_vendor
  -- in preparation for update or insert
  CASE company_type_in
    WHEN 'staff' THEN
      l_staff := '1';
    WHEN 'client' THEN
      l_client := '1';
    WHEN 'vendor' THEN
      l_vendor := '1';
    ELSE
      RAISE no_such_company_type;
  END CASE;
  
  -- If l_company_id is NULL, it means this is a new company.
  IF l_company_id IS NULL THEN
    INSERT INTO companies
    (company_name, staff, client, vendor)
    VALUES(company_name_in, l_staff, l_client, l_vendor)
    RETURNING company_id INTO l_company_id;
  -- Otherwise, a company with this name already existed
  -- in the companies table, so we update.
  ELSE
    UPDATE companies
    SET staff   = l_staff,
        client  = l_client,
        vendor  = l_vendor
    WHERE company_name = company_name_in;
  END IF;
  
  -- Insert new person into people
  INSERT INTO people
  (first_name, last_name, email, phone_number, company_id)
  VALUES
  (first_name_in,last_name_in,email_in,phone_number_in,l_company_id)
  RETURNING person_id INTO l_staff_person_id;
  
  -- If new person is staff, insert into people_employees. Really,
  -- we'd have IN parameters for all fields in people_employees.
  IF company_type_in = 'staff' THEN
    INSERT INTO people_employees
    (person_id, hire_date)
    VALUES (l_staff_person_id, hire_date_in);
  END IF;

EXCEPTION
  WHEN no_such_company_type THEN
    RAISE_APPLICATION_ERROR(-20101,
                  'Company type must be staff, client, or vendor.');
  
END insert_person;