/******************************************************************************
  Global variables
******************************************************************************/
let high; // The highest number used in equations.
let low = 0; // The lowest number used in equations.
let operator; // The operator currently being used.


/******************************************************************************
  Utility Functions
******************************************************************************/
function randInt(low, high) {
  /* Utility function for getting random integer. */
  return Math.floor(Math.random() * (high - low + 1) + low);
}


/******************************************************************************
  Common Functions
******************************************************************************/
function captureChangeViewEvents() {
  // Cache reused elements
  const configContainer = document.getElementById('config-container');
  const gameContainer = document.getElementById('game-container');
  const timesUpContainer = document.getElementById('times-up-container');

  // Capture play button clicks
  const playButton = document.getElementById('btn-play-button');
  playButton.addEventListener('click', function() {
    // Set global variable: high
    high = Number(document.getElementById('max-number').value);

    // Set global variable: operator
    switch (document.getElementById('operation').value) {
      case 'addition':
        operator = '+';
        break;
      case 'subtraction':
        operator = '-';
        break;
      case 'multiplication':
        operator = 'x';
        break;
      case 'division':
        operator = '/';
    }
    switchSection(configContainer, gameContainer);
    play();
  });

  // Capture play-again button clicks
  const playAgainButton = document.getElementById('btn-play-again');
  playAgainButton.addEventListener('click', function() {
    switchSection(timesUpContainer, gameContainer);
    play();
  });

  // Capture change-settings button clicks
  const changeSettingsButton = document.getElementById('btn-change-settings');
  changeSettingsButton.addEventListener('click', function() {
    switchSection(timesUpContainer, configContainer);
  });
}

function switchSection(hideSection, showSection) {
  // Hide one section and display another
  hideSection.style.display = 'none';
  showSection.style.display = 'block';

  // If leaving game, turn off game event capturing
  if (hideSection.id === 'game-container') {
    captureGameEvents('off');
  }

  // If starting game, turn on game event capturing
  if (showSection.id === 'game-container') {
    captureGameEvents('on');
  }
}

/******************************************************************************
  Config View Functions
******************************************************************************/
function populateSelects() {
  // Cache reused elements
  const operationSelect = document.getElementById('operation');
  const maxNumberSelect = document.getElementById('max-number');

  // Populate operator options
  const operations = ['Addition', 'Subtraction', 'Multiplication', 'Division'];
  
  operations.forEach((operation) => {
    const option = document.createElement('option');
    option.setAttribute('value', operation.toLowerCase());
    option.innerHTML = operation;
    operationSelect.appendChild(option);
  })

  operationSelect.value = 'multiplication'; // Initial operator
  operationSelect.focus();

  // Populate max number options
  for (let i = 2; i <= 100; i++) {
    const option = document.createElement('option');
    option.setAttribute('value', i);
    option.innerHTML = i;
    maxNumberSelect.appendChild(option);
  }

  maxNumberSelect.value = 10; // Initial Max Number
}

/******************************************************************************
  Game View Functions
******************************************************************************/
function answerChanged(value) {
  /* 
    This is the callback function that gets called each time the user adds
    their answer. It gets the user's current answer, appends the new change,
    checks the result against the correct answer. If correct, it calls
    setEquation() to get a new equation.
  */
  // Cache reused elements
  const userSolutionElem = document.getElementById('user-solution');
  const scoreElem = document.getElementById('score');

  const correctAnswer = getCorrectAnswer(operator);
  let score = Number(scoreElem.innerText);
  let userAnswer = Number(userSolutionElem.innerText);

  if (userAnswer === 0) {
    // If userAnswer is 0, don't append, just change the value.
    // This prevents answers like "05"
    userSolutionElem.innerText = value;
  } else {
    // Append the new value to the user's previous answer.
    userSolutionElem.innerText += value;
  }

  // Retrieve and convert the new answer to an integer.
  userAnswer = Number(userSolutionElem.innerText);

  // Check if the answer is correct
  if (correctAnswer === userAnswer) {
    score++;
    scoreElem.innerText = score.toString(); // Update scoreboard.
    userSolutionElem.innerText = ''; // Clear user answer.
    setEquation(operator, low, high); // Get new equation.
  };
}

function captureGameEvents(onoff = 'on') {
  /*
    This is used to turn event capturing for game-play events on and off. Its
    primary purpose is for keyboard events, as the use can continue to press
    keys after the game view has disappeared.
  */
  const numPadButtons = document.getElementsByClassName('number-button');

  // Decide whether we are adding or removing events.
  const eventAction = (onoff === 'on') ?
    'addEventListener' : 'removeEventListener';

  // Add/Remove click event to all number buttons
  for (numBtn of numPadButtons) {
    numBtn[eventAction]('click', numButtonClicked);
  }

  // Add/Remove click event to clear button
  const clearButton = document.getElementById('btn-clear-button');
  clearButton[eventAction]('click', clearButtonClicked);

  // Add/Remove keyup events to enable keyboard play
  document[eventAction]('keyup', keyPressed);
}

function clearButtonClicked() {
  document.getElementById('user-solution').innerText = '';
}

function getCorrectAnswer(oper) {
  /* Calculate and return correct answer. */
  const num1 = Number(document.getElementById('num1').innerText);
  const num2 = Number(document.getElementById('num2').innerText);
  switch (oper) {
    case '+':
      return num1 + num2;
    case '-':
      return num1 - num2;
    case 'x':
      return num1 * num2;
    case '/':
      return num1 / num2;
  }
}

function keyPressed(e) {
  /* Callback function to keyboard clicks. */
  // Cache reused elements
  const userSolutionElem = document.getElementById('user-solution');

  const value = e.key;
  if (value === 'Backspace' || value === 'Delete') {
    const currentAnswer = userSolutionElem.innerText;
    if (currentAnswer.length) {
      const newAnswer = currentAnswer.substring(0, currentAnswer.length - 1);
      userSolutionElem.innerText = newAnswer;
    }
  } else if (value === ' ') { // spacebar
    userSolutionElem.innerText = '';
  } else if (!isNaN(value)) { // key is a number key
    answerChanged(value);
  }
}

function numButtonClicked(e) {
  /* Callback function to number button clicks. */
  const btn = e.target; // Which button?
  const value = btn.dataset.value; // Get the value of the clicked button.
  answerChanged(value); // Change the answer.
}

function play() {
  /* Start playing. */
  document.getElementById('time-left').innerText = 60; // seconds left
  document.getElementById('score').innerText = 0;
  setEquation();
  startTimer();
}

function setEquation() {
  /* Sets the equation. */
  // Cache reused elements
  const userSolutionElem = document.getElementById('user-solution');
  const equationElem = document.getElementById('equation');

  let num1 = randInt(low, high);
  let num2 = randInt(low, high, true);
  if (operator === '-') { // Subtract smaller number from bigger number.
    num1 = Math.max(num1, num2);
    num2 = Math.min(num1, num2);
  } else if (operator === '/') {
    while (num2 === 0) { // No division by zero
      num2 = randInt(low, high);
    }
    // Make num1 the product of num1 x num2, so that the equation is num1/num2.
    // e.g., if the random numbers are 5 and 7, the equation will be 35 / 7.
    num1 = num1 * num2;
  }

  // Create the spans to hold the operands
  const spanNum1 = '<span id="num1">' + num1.toString() + '</span>';
  const spanNum2 = '<span id="num2">' + num2.toString() + '</span>';

  // Clear the user answer.
  userSolutionElem.innerText = '';

  equationElem.innerHTML = spanNum1 + ' ' + operator + ' ' + spanNum2;
}

function startTimer() {
  // Start the Timer Countdown.
  const timer = setInterval(function() {
    const timeLeft = document.getElementById('time-left');

    let secondsLeft = Number(timeLeft.innerText);
    secondsLeft--;
    timeLeft.innerText = secondsLeft.toString();

    if (secondsLeft < 10) { // Start changing background color.
      const a = (10 - secondsLeft) * .1;
      const bgColor = 'rgb(255,204,102,' + a + ')';
      document.body.style.backgroundColor = bgColor;
    }

    if (secondsLeft === 0) { // Times up.
      const score = Number(document.getElementById('score').innerText);
      clearInterval(timer);
      timesUp(score);
    }
  }, 1000);
}

function timesUp(score) {
  document.body.style.backgroundColor = '#fff'; // Reset background to white.

  // Show the times-up view.
  switchSection(
    document.getElementById('game-container'),
    document.getElementById('times-up-container')
  );

  // Show the score.
  document.getElementById('final-score').innerText = score;
}

/******************************************************************************
  Initial Load
******************************************************************************/
$(function() {
  populateSelects();
  captureChangeViewEvents();
});