#include <iostream>
#include <string>
#include <vector>
#include <memory>

// Base class: Person
class Person {
private:
    int id_;               // hidden (encapsulation)
    std::string name_;     // hidden

public:
    Person(int id, const std::string& name) : id_(id), name_(name) {}
    virtual ~Person() = default; // virtual destructor for correct cleanup

    // getters and setters
    int getId() const { return id_; }
    void setId(int id) { id_ = id; }

    std::string getName() const { return name_; }
    void setName(const std::string& name) { name_ = name; }

    // Polymorphic behavior: derived classes override this
    virtual void howPaid() const {
        std::cout << "Person payment: unspecified\n";
    }

    // Another polymorphic function
    virtual void display() const {
        std::cout << "Person: ID=" << id_ << ", Name=" << name_ << "\n";
    }
};
// Derived class: Employee
// Inheritance
class Employee : public Person {
private:
    double salary_;        // encapsulated salary

public:
    Employee(int id, const std::string& name, double salary)
        : Person(id, name), salary_(salary) {}

    // Employee-specific getter/setter
    double getSalary() const { return salary_; }
    void setSalary(double s) { salary_ = s; }

    // Polymorphism
    void howPaid() const override {
        std::cout << "Employee paid monthly salary: $" << salary_ << "\n";
    }

    // Polymorphism
    void display() const override {
        std::cout << "Employee: ID=" << getId()
                  << ", Name=" << getName()
                  << ", Salary=$" << salary_ << "\n";
    }
};
// Derived class: Contractor
// Inheritance
class Contractor : public Person {
private:
    std::string company_;  // contracting company (encapsulated)
    double hourlyRate_;

public:
    Contractor(int id, const std::string& name, const std::string& company, double rate)
        : Person(id, name), company_(company), hourlyRate_(rate) {}

    std::string getCompany() const { return company_; }
    void setCompany(const std::string& c) { company_ = c; }

    double getHourlyRate() const { return hourlyRate_; }
    void setHourlyRate(double r) { hourlyRate_ = r; }

    // Polymorphism
    void howPaid() const override {
        std::cout << "Contractor paid by company '" << company_
                  << "' at hourly rate $" << hourlyRate_ << "\n";
    }

    // Polymorphism
    void display() const override {
        std::cout << "Contractor: ID=" << getId()
                  << ", Name=" << getName()
                  << ", Company=" << company_
                  << ", Rate=$" << hourlyRate_ << "/hr\n";
    }
};

int main() {
    std::vector<std::unique_ptr<Person>> staff;

    // Create some employees and contractors (encapsulation via constructors + setters)
    staff.push_back(std::make_unique<Employee>(101, "Alice Smith", 5500.0));
    staff.push_back(std::make_unique<Contractor>(201, "Bob Jones", "BuildCo", 45.0));
    staff.push_back(std::make_unique<Employee>(102, "Charlie King", 4800.0));
    staff.push_back(std::make_unique<Contractor>(202, "Dana Lee", "DesignLLC", 60.0));

    // Demonstrate iteration, polymorphic calls, and encapsulated access
    std::cout << "--- Staff Directory (polymorphic display) ---\n";
    for (const auto& p : staff) {
        p->display();        // runs derived class version
    }

    std::cout << "\n--- How each person is paid (polymorphic howPaid) ---\n";
    for (const auto& p : staff) {
        p->howPaid();        // runs derived class version
    }

    // Show encapsulation: using getters and setters
    std::cout << "\n--- Updating and showing encapsulated data ---\n";

    // Give a raise to Alice (employee at index 0)
    if (auto emp = dynamic_cast<Employee*>(staff[0].get())) {
        double old = emp->getSalary();
        emp->setSalary(old + 500.0); // safe update via setter
        std::cout << "Alice received a raise. New salary: $" << emp->getSalary() << "\n";
    }

    // Change contractor's company (index 1)
    if (auto con = dynamic_cast<Contractor*>(staff[1].get())) {
        con->setCompany("RenovateCo");
        std::cout << "Bob's new company: " << con->getCompany() << "\n";
    }

    return 0;
}
