#include <iostream>
#include <format>
#include <stdexcept>
#include <string>
#include <vector>
#include <deque>
#include <map>
#include <algorithm>

typedef std::string str;

const double START_BALANCE = 0.0; // Initial account balance
str userInput;

class Account {
public: 
    enum class AcctType {
        Checking,
        Savings
    };

private:
    int acctNbr;
    str holder;
    double balance;
    AcctType acctType;

public:
    // Constructor with all variables
    Account(int acctNbr, str holder, int balance, AcctType acctType)
        : acctNbr(acctNbr), holder(holder), balance(balance), acctType(acctType) {
    }
    // default constructor (no parameters)
    Account() {
        acctNbr = 999999999;
        holder = "N/A";
        balance = 0.0;
    }
    // Getters
    int getAcctNbr()  { return acctNbr; }
    str getHolder()  { return holder; }
    double getBalance() { return balance; }
    AcctType getAcctType() { return acctType; }

    // Setters
    void setAcctNbr(int acctNbr) { this->acctNbr = acctNbr; }
    void setHolder(str holder) { this->holder = holder; }
    void setBalance(double balance) { this->balance = balance; }
    void setAcctType(AcctType acctType) { this->acctType = acctType; }

};

str displayAcctType(Account::AcctType acctType) {
    switch (acctType) {
    case Account::AcctType::Checking:
        return "Checking";
    case Account::AcctType::Savings:
        return "Savings";
    default:
        return "Invalid Account Type";
    }
}

void printAccountObject(Account account) {
    std::cout << std::format("{:^70}\n", "Bank Account Report");
    std::cout << std::format("{:<15} {:<20}\n", "Account Number", account.getAcctNbr());
    std::cout << std::format("{:<15} {:<20}\n", "Account Holder", account.getHolder());
    std::cout << std::format("{:<15} {:<20}\n", "Account Type", displayAcctType(account.getAcctType()));
    std::cout << std::format("{:<15} ${:<20.2f}\n", "Balance", account.getBalance());
}

double withdraw(double amount, double balance) {
    static int nbrOfWithdrawals = 1;
    if (amount > balance) {
        std::string error_msg = std::format("Withdrawal rejected: amount ${:.2f} exceeds balance.\n", amount);
        throw std::runtime_error(error_msg);
    }
    else if (amount <= 0) {
        std::string error_msg = std::format("Withdrawal rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
        throw std::runtime_error(error_msg);
    }
    else {
        std::cout << std::format("Number of withdrawals is now {}\n", nbrOfWithdrawals);
        if (nbrOfWithdrawals > 3) {
            amount += 3;
        }
        nbrOfWithdrawals++;
        return balance -= amount;
    }
}

double deposit(double amount, double balance) {
    if (amount <= 0) {
        std::string error_msg = std::format("Deposit rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
        throw std::runtime_error(error_msg);
    }
    else {
        return balance += amount;
    }
}

int main() {

    str userInput;

    std::vector<double> depositLog{};  // use a vector for deposits
    std::deque<double> withdrawDeque; // use deque for withdrawals
    std::map<std::string, double> transactionsMap;
    // set total deposits to 0
    transactionsMap["Deposits"] = 0;
    // set total withdrawals to 0
    transactionsMap["Withdrawals"] = 0;

    // Call default constructor and use setter to set account number:
    Account account = Account();
    account.setAcctNbr(555666777);
    // prompt user for account holder name:
    std::cout << "Please enter account holder name: ";
    getline(std::cin, userInput);
    account.setHolder(userInput);

    bool correct = false;
    // prompt user for account type:
    while (!correct && userInput[0] != 'Q') {
        std::cout << "Enter account type (C for Checking, S for Savings) or Q to Quit the app: ";
        getline(std::cin, userInput);
        switch (userInput[0]) {
        case 'C':
        case 'c':
            account.setAcctType(Account::AcctType::Checking);
            correct = true;
            break;
        case 'S':
        case 's':
            account.setAcctType(Account::AcctType::Savings);
            correct = true;
            break;
        case 'Q':
        case 'q':
            std::cout << "Closing the app by user request.\n";
            return 0;
        default:
            std::cout << "Unknown account type...please re-enter (Q to quit).\n";
        }
    }

    do {
        // Display menu options
        std::cout << "\n Deposit (d)\n";
        std::cout << " Withdraw (w)\n";
        std::cout << " Balance (b)\n";
        std::cout << " Quit (q)\n\n";
        std::cout << "Enter choice: ";
        std::getline(std::cin, userInput);
        switch (userInput[0]) {
            // Handle deposit
            case 'd':
            case 'D': {
                double amount;
                std::cout << "Enter amount to deposit: ";
                std::getline(std::cin, userInput);
                try {
                    amount = std::stod(userInput);
                    account.setBalance(deposit(amount, account.getBalance()));
                    // log deposit amount to vector
                    depositLog.push_back(amount);  
                    // add deposit amount to running total on the map 
                    transactionsMap["Deposits"] += amount; 
                    std::cout << std::format("Deposited ${:.2f}\n", amount);
                }
                catch (std::invalid_argument& ie) {
                   std::cout <<  "Deposit amount must be numeric, amount rejected.\n";
                }
                catch (std::runtime_error& re) {
                   std::cout << re.what();
                }  
                break;
            }
            // Handle withdrawal
            case 'w':
            case 'W': {
                double amount;
                std::cout << "Enter withdrawal amount: ";
                std::getline(std::cin, userInput);
                try {
                    amount = std::stod(userInput);
                    account.setBalance(withdraw(amount, account.getBalance()));
                    // log withdrawal to deque
                    // use push_front so that most recent withdrawal is front of the deque
                    withdrawDeque.push_front(amount);   
                    // add withdrawal amount to running total on the map 
                    transactionsMap["Withdrawals"] += amount;  
                    std::cout << std::format("Withdrew ${:.2f}\n", amount);
                }
                catch (std::invalid_argument& ie) {
                   std::cout <<  "Withdrawal amount must be numeric, amount rejected.\n";
                }
                catch (std::runtime_error& rte) {
                    std::cout << rte.what();
                }    
                break;
            }
            // Display balance
            case 'b':
            case 'B':
                std::cout << std::format("Current balance: ${:.2f}\n", account.getBalance());
                break;
            // Handle quit
            case 'q':
            case 'Q':
                std::cout << std::format("Final balance: ${:.2f}\n", account.getBalance());
                break;
            // Handle invalid input
            default:
                std::cout << "Invalid selection. Please choose a valid option.\n";
        }
    } while (userInput[0] != 'q' && userInput[0] != 'Q');

    // print withdrawDeque as a stack so more recent withdrawals are displayed first
    std::cout << std::format("{:^20}\n", "Log of Withdrawals-Most Recent First");
    for (const auto& wd : withdrawDeque) {
        std::cout << std::format("{:>20}\n", std::format("${:.2f}", wd));
    }
    // print withdrawDeque as a queue so least recent withdrawals are displayed first
    std::cout << std::format("{:^20}\n", "Log of Withdrawals-Least Recent First");
    for (auto wdIter = withdrawDeque.rbegin(); wdIter != withdrawDeque.rend(); wdIter++) {
        std::cout << std::format("{:>20}\n", std::format("${:.2f}", *wdIter));
    }
    // print deposit log
    std::cout << std::format("{:^20}\n", "Log of Deposits");
    std::sort(depositLog.begin(), depositLog.end(), std::greater<>{});  // sort depositss in descending sequence
    for (auto& deposit : depositLog) {
        if (deposit == 0) {
            break;
        }
        std::cout << std::format("{:>20}\n", std::format("${:.2f}", deposit));
    }
    // print account object
    printAccountObject(account);
    // print totals from transactions map
    std::cout << std::format("Total amount of deposits is ${:.2f}.\n", transactionsMap["Deposits"]);
    std::cout << std::format("Total amount of withdrawals is ${:.2f}.\n", transactionsMap["Withdrawals"]);
}