#include <iostream>
#include <format>
#include <vector>
#include <memory>

class Person {
private:
    std::string firstName;
    std::string lastName;

public:
    // Constructor with first name and last name
    Person(const std::string& fName, const std::string& lName)
        : firstName(fName), lastName(lName) {
    }
    // Overloaded constructor with last name only
    Person(const std::string& lName)
        :lastName(lName) {
        setFirstName("");
    }

    // Getters
    std::string getFirstName() const { return firstName; }
    std::string getLastName() const { return lastName; }

    // Setters
    void setFirstName(const std::string& fName) { firstName = fName; }
    void setLastName(const std::string& lName) { lastName = lName; }

    // Destructor
    ~Person() {}

    // virtual function
    virtual void display() {
        std::cout << std::format("Person with name {} {}.\n", getFirstName(), getLastName());
    }
    // pure virtual function (must be implemented in sublclasses)
    virtual std::string howPaid() = 0;

    // overload == operator
    bool operator==(Person& anotherPerson) {
        return (firstName == anotherPerson.firstName && lastName == anotherPerson.lastName);
    }
};

// create Employee subclass
class Employee : public Person {
private:
    int empId;

public:
    Employee(const std::string& fName, const std::string& lName, int id)
        : Person(fName, lName), empId(id) {
    }
    Employee(const std::string& lName, int id)
        : Person(lName), empId(id) {
    }

    void setEmpId(int id) { empId = id; }
    int getEmpId() const { return empId; }

    void display()  override {   // override Person class display()
        Person::display();  // call Person class display()
        std::cout << std::format("An employee with employee ID {}.\n", getEmpId()); // add Employee ID
    }

    std::string howPaid() override {  // must be overriden here in subclass
        return "by W-2";
    }
};

// create Contractor subclass
class Contractor : public Person {
private:
    std::string contractorCompany;

public:
    Contractor(const std::string& fName, const std::string& lName, const std::string& company)
        : Person(fName, lName), contractorCompany(company) {
    }

    void setContractorCompany(const std::string& company) { contractorCompany = company; }
    std::string getContractorCompany() const { return contractorCompany; }

    void display()  override {  // override Person class display()
        Person::display(); // call Person class display()
        // add Contractor Company:
        std::cout << std::format("A contractor working for {}.\n", getContractorCompany());
    }

    std::string howPaid() override { // must be overriden here in subclass
        return "by 1099";
    }
};

int main() {
    // create an Employee
    Employee joeSmith("Joe", "Smith", 112233);
    joeSmith.display();

    // create a Contractor
    Contractor jenniferJones("Jennifer", "Jones", "West Coast IT Consulting LLC");
    jenniferJones.display();

    // demonstrate polymorphic behaviour of display()
    // first, create a vector of unique_ptr and type to Person (base class):
    std::vector<std::unique_ptr<Person>> persons;
    
    persons.push_back(std::make_unique<Employee>(joeSmith)); // add Employee object to vector
    persons.push_back(std::make_unique<Contractor>(jenniferJones)); // add Contractor object to vector

    // use auto to adapt to the object type in the for loop
    std::cout << "\nPolymorphsim demo: Iterate the vector and call display on each object\n";
    for (auto& person : persons) {
        person->display();      // the correct display function is called at runtime
        // uncomment if you want to dereference the unique_ptr so that you an use dot notation to call display()
        //(*person).display();
        std::cout << std::format("Paid {}.\n", person->howPaid());
    }
}
