#include <iostream>
#include <format>
#include <stdexcept>
#include <string>
#include <vector>
#include <map>
#include <algorithm>
#include <deque>
#include <fstream>
#include <memory>

typedef std::string str;

const double START_BALANCE = 0.0; 

str userInput;

class Account {

private:
    int acctNbr;
    str holder;
    double balance;

protected:
    void setBalance(double balance) { this->balance = balance; }

public:
    Account(int acctNbr, str holder, double balance)
        : acctNbr(acctNbr), holder(holder), balance(balance) {
    }
    // default constructor (no parameters)
    Account() {
        acctNbr = 999999999;
        holder = "N/A";
        balance = 0.0;
    }
    // Getters
    int getAcctNbr() { return acctNbr; }
    str getHolder() { return holder; }
    double getBalance() { return balance; }

    // Setters
    void setAcctNbr(int acctNbr) { this->acctNbr = acctNbr; }
    void setHolder(str holder) { this->holder = holder; }

    // virtual display function
    virtual std::string display() {
        return std::format("Account number {} with holder {} and balance of ${:.2f}.\n",
            getAcctNbr(), getHolder(), getBalance());
    }

    //pure virtual functions
    virtual void deposit(double deposit) = 0;
   
    virtual void withdraw(double withdrawal) = 0;
    
};

class Checking : public Account {
private:
    bool paperChecks;
public:
    Checking(int acctNbr, str holder, double balance, bool paperChecks)
        : Account(acctNbr, holder, balance), paperChecks(paperChecks) {
    }
    void deposit(double amount) override {
        if (amount <= 0) {
            std::string error_msg = 
                std::format("Deposit rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
            throw std::runtime_error(error_msg);
        }
        double balance = getBalance();
        setBalance(balance + amount);
    }
    void withdraw(double amount) override {
        if (amount <= 0) {
            std::string error_msg = 
                std::format("Withdrawal rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
            throw std::runtime_error(error_msg);
        }
        double balance = Account::getBalance();
        Account::setBalance(balance - amount);
    }
    // virtual display function
    virtual std::string display() {
        return Account::display() + std::format("Paper checks for this Checking account is {}.\n", paperChecks);
    }
};

class Savings : public Account {
private:
    double transactionFee;
    // track number of withdrawals for each Savings object:
    int nbrOfWithdrawals = 0;
public:
    // Transaction fee for a Savings withdrawal
    const double SAVINGS_TRANSACTION_FEE = 2.50; 
    // Fee for exceeding maximum number of Savings withdrawals:
    const double SAVINGS_MAX_WD_FEE = 3.00; 
    // Maximum number of savings withdrawals without fee:
    const int MAX_SAVINGS_WITHDRAWALS = 3;

    // Constructor with transaction fee
    Savings(int acctNbr, str holder, double balance, double transactionFee)
        : Account(acctNbr, holder, balance), transactionFee(transactionFee) {
    }
    // Constructor without transaction fee...default transaction fee will be used
    Savings(int acctNbr, str holder, double balance) 
        : Account(acctNbr, holder, balance) {
            transactionFee = SAVINGS_TRANSACTION_FEE;
    }

    void deposit(double amount) override {
        if (amount <= 0) {
            std::string error_msg = 
                std::format("Deposit rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
            throw std::runtime_error(error_msg);
        }
        double balance = Account::getBalance();
        setBalance(balance + amount);
    }
    void withdraw(double amount) override {
        if (amount > Account::getBalance()) {
            throw std::runtime_error("Withdrawal amount exceeds balance");
        }
        else if (amount <= 0) {
            std::string error_msg = 
                std::format("Withdrawal rejected: amount ${:.2f} is less than or equal to zero.\n", amount);
            throw std::runtime_error(error_msg);
        }
        else {
            nbrOfWithdrawals++;
            if (nbrOfWithdrawals > MAX_SAVINGS_WITHDRAWALS) {
                amount += SAVINGS_MAX_WD_FEE;
            }
    }
        double balance = Account::getBalance() - transactionFee;
        setBalance(balance - amount);
    }
    // virtual display function
    virtual std::string display() {
        return Account::display() + std::format("Transaction fee for this Savings account is ${:.2f}.\n", transactionFee);
    }
};

int main() {

    std::vector<double> depositLog{}; // use vector for deposits
    std::deque<double> withdrawDeque; // use deque for withdrawals
    std::map<std::string, double> transactionsMap;
    // set total deposits to 0
    transactionsMap["Deposits"] = 0;
    // set total withdrawals to 0
    transactionsMap["Withdrawals"] = 0;

    // prompt user for account holder name:
    std::cout << "Please enter account holder name: ";
    getline(std::cin, userInput);
    std::string accountHolder = userInput;
    // create Account object pointer:
    std::unique_ptr<Account> account;

    bool correct = false;
    // prompt the user for account type:
    while (!correct && userInput[0] != 'Q') {
        bool continueLooping = true; // used in the 'S' case of the switch statement
        std::cout << "Enter account type (C for Checking, S for Savings) or Q to Quit the app:\n";
        getline(std::cin, userInput);
        switch (userInput[0]) {
        case 'C':
        case 'c':
            std::cout << "Do you want paper checks (Y/N)? ";
            getline(std::cin, userInput);
            if (userInput[0] == 'Y' || userInput[0] == 'y') {
                account = std::make_unique<Checking>(555666777, accountHolder, START_BALANCE,  true);
            }
            else {
                account = std::make_unique<Checking>(555666777, accountHolder, START_BALANCE, false);
            }
            correct = true;
            break;
        case 'S':
        case 's': 
             do { 
                std::cout << "Enter the transaction fee for each withdrawal or \'d\' for the default fee ($2.50): ";
                try {
                    getline(std::cin, userInput);
                    if (userInput[0] == 'd' || userInput[0] == 'D') {  
                        // user wants to use the default fee
                        account = std::make_unique<Savings>(555666777, accountHolder, START_BALANCE);  
                    }
                    else {
                        double userTransactionFee = std::stod(userInput);  // get user transaction fee
                        if (userTransactionFee < 0) {
                            // throw invalid argument exception with message place holder
                            // (hard coded message will be displayed in catch block)
                            throw std::invalid_argument("message");
                        }
                        account = std::make_unique<Savings>(555666777, accountHolder, START_BALANCE, userTransactionFee);
                    }
                    continueLooping = false;
                }
                catch (std::invalid_argument& ie) {
                    std::cout << "You entered a non-numeric or negative fee. Please try again.\n";
                }
            }
            while (continueLooping);
            correct = true;
            break;
        case 'Q':
        case 'q':
            std::cout << "Closing the app by user request.\n";
            return 0;
        default:
            std::cout << "Unknown account type...please re-enter (Q to quit).\n";
        }
    }

    do {
        // Display menu options
        std::cout << " Deposit (d)\n";
        std::cout << " Withdraw (w)\n";
        std::cout << " Balance (b)\n";
        std::cout << " Quit (q)\n\n";
        std::cout << "Enter choice: ";
        std::getline(std::cin, userInput);
        switch (userInput[0]) {
            // Handle deposit
            case 'd':
            case 'D': {
                double amount;
                std::cout << "Enter amount to deposit: ";
                std::getline(std::cin, userInput);
                try {
                    amount = std::stod(userInput);
                    account->deposit(amount);
                    // log deposit amount to vector
                    depositLog.push_back(amount);  
                    // add deposit amount to running total on the map 
                    transactionsMap["Deposits"] += amount; 
                    std::cout << std::format("Deposited ${:.2f}\n", amount);
                }
                catch (std::invalid_argument& ie) {
                   std::cout <<  "Deposit amount must be numeric. Please try again.\n";
                }
                catch (std::runtime_error& re) {
                   std::cout << re.what();
                } 
                break;
            }
            // Handle withdrawal
            case 'w':
            case 'W': {
                double amount;
                std::cout << "Enter withdrawal amount: ";
                std::getline(std::cin, userInput);
                try {
                    amount = std::stod(userInput);
                    account->withdraw(amount);
                    // log withdrawal to deque
                    // use push_front so that most recent withdrawal is front of the deque
                    withdrawDeque.push_front(amount);    
                    // add withdrawal amount to running total on the map 
                    transactionsMap["Withdrawals"] += amount;  
                    std::cout << std::format("Withdrew ${:.2f}\n", amount);
                }
                catch (std::invalid_argument& ie) {
                   std::cout <<  "Withdrawal amount must be numeric, amount rejected.\n";
                }
                catch (std::runtime_error rte) {
                    std::cout << rte.what();
                }   
                break;
            }
            // Display balance
            case 'b':
            case 'B':
                std::cout << std::format("Current balance: ${:.2f}\n", account->getBalance());
                break;
            // Handle quit
            case 'q':
            case 'Q':
                std::cout << std::format("Final balance: ${:.2f}\n", account->getBalance());
                break;
            // Handle invalid input
            default:
                std::cout << "Invalid selection. Please choose a valid option.\n";
        }
    } while (userInput[0] != 'q' && userInput[0] != 'Q');

    // open Account Log file
    std::ofstream accountLog("AccountLog.txt");
    // write account information to account file
    accountLog << std::format("{}", account->display());
    accountLog << std::format("Total amount of deposits is ${:.2f}.\n", transactionsMap["Deposits"]);
    accountLog << std::format("Total amount of withdrawals is ${:.2f}.\n", transactionsMap["Withdrawals"]);
    // write withdrawal log to a file - treat deque as a stack
    accountLog << std::format("{:^20}\n", "Log of Withdrawals - Most Recent First");
    for (const auto& wd : withdrawDeque) {
        accountLog << std::format("{:>20}\n", std::format("${:.2f}", wd));
    }
    // print withdrawDeque as a queue so least recent withdrawals are displayed first
    accountLog << std::format("{:^20}\n", "Log of Withdrawals-Least Recent First");
    for (auto wdIter = withdrawDeque.rbegin(); wdIter != withdrawDeque.rend(); wdIter++) {
        accountLog << std::format("{:>20}\n", std::format("${:.2f}", *wdIter));
    }
    // write deposit log to a file
    accountLog << std::format("{:^20}\n", "Log of Deposits");
    std::sort(depositLog.begin(), depositLog.end(), std::greater<>{});  // sort deposits in descending sequence
    for (auto& deposit : depositLog) {
        if (deposit == 0) {
            break;
        }
        accountLog << std::format("{:>20}\n", std::format("${:.2f}", deposit));
    } 
    accountLog.close();
    std::cout << "Account log available in AccountLog.txt.\n";
    std::cout << "Do you want to view the account log?\n";
    getline(std::cin, userInput);  
    if (userInput[0] == 'Y' || userInput[0] == 'y') {
        std::ifstream accountLog("AccountLog.txt");
        std::string record;
        if (accountLog.is_open()) {
            while (getline(accountLog, record)) {
                std::cout << record << "\n";
            }
        }
        else {
            std::cout << "Could not open the account log file.\n";
        }
        accountLog.close();
    }
}