#include <iostream>
#include <fstream>   // Needed for ifstream, ofstream, and fstream
#include <string>

int main() {

    // Part 1: Opening a file in write mode (std::ios::out). This will CREATE the file if it doesn't exist. If it DOES exist, its content will be ERASED (truncated).
    std::ofstream writeFile("example.txt", std::ios::out);

    if (!writeFile) {
        std::cout << "Error: Could not open example.txt for writing.\n";
        return 1;   // Exit if opening failed
    }

    writeFile << "This is a test line.\n";
    writeFile << "Writing to a file using ios::out mode.\n";

    writeFile.close();   // Always close the file
    std::cout << "Write operation complete.\n\n";

    // Part 2: Opening the same file in append mode (std::ios::app). This mode does NOT erase existing content. New text is added to the END of the file.
    std::ofstream appendFile("example.txt", std::ios::app);

    if (!appendFile) {
        std::cout << "Error: Could not open example.txt for appending.\n";
        return 1;
    }

    appendFile << "This line was added using append mode.\n";

    appendFile.close();
    std::cout << "Append operation complete.\n\n";

    // Part 3: Reading from the file using ifstream. We add error handling and end-of-file detection.
    std::ifstream readFile("example.txt", std::ios::in);

    if (!readFile) {
        std::cout << "Error: Could not open example.txt for reading.\n";
        return 1;
    }

    std::cout << "Contents of example.txt:\n";

    std::string line;

    // Read the file line by line until EOF is reached
    while (std::getline(readFile, line)) {
        std::cout << line << std::endl;
    }

    // Check why the loop ended
    if (readFile.eof()) {
        std::cout << "\nReached end of file.\n";
    } else if (readFile.fail()) {
        std::cout << "\nError: A read operation failed.\n";
    }

    readFile.close();
    std::cout << "Read operation complete.\n\n";

    // Part 4: Using fstream for both reading and writing. We open with: std::ios::in  -> read and std::ios::out -> write. This requires the file to already exist.
    std::fstream file("example.txt", std::ios::in | std::ios::out);

    if (!file) {
        std::cout << "Error: Could not open example.txt for read/write.\n";
        return 1;
    }

    // Write a new line at the end
    file.clear();               // Clear any EOF flags
    file.seekp(0, std::ios::end);  // Move write position to the end
    file << "Added using fstream read/write mode.\n";

    file.close();
    std::cout << "Combined read/write operation complete.\n";

    return 0;
}